﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Hims.Api.Utilities;
    using Hims.Domain.Services;
    using Hims.Shared.DataFilters;
    using Hims.Shared.EntityModels;
    using Hims.Shared.UserModels.Filters;

    // ReSharper disable StyleCop.SA1126

    /// <summary>
    /// The wallet controller.
    /// </summary>
    [Route("api/wallet")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class WalletController : BaseController
    {
        /// <summary>
        /// the wallet service
        /// </summary>
        private readonly IWalletService walletServices;

        /// <inheritdoc />
        public WalletController(IWalletService walletServices) => this.walletServices = walletServices;

        /// <summary>
        /// To fetch audit logs
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The list of audit logs.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of auditLogs.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("fetch")]
        [ProducesResponseType(typeof(IEnumerable<WalletFilterModel>), 200)]
        [ProducesResponseType(500)]

        public async Task<ActionResult> FetchAsync([FromBody] WalletFilterModel model)
        {
            model = (WalletFilterModel)EmptyFilter.Handler(model);
            var wallets = await this.walletServices.FetchAsync(model);
            if (wallets == null)
            {
                return ServerError();
            }

            foreach (var wallet in wallets)
            {
                wallet.TotalAvailable = wallet.TotalCredits - wallet.TotalDebits;
            }

            return Success(wallets);
        }

        /// <summary>
        /// To insert audit logs
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The list of audit logs.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of auditLogs.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("create")]
        [ProducesResponseType(500)]
        public async Task<int> InsertAsync([FromBody] WalletModel model)
        {
            model = (WalletModel)EmptyFilter.Handler(model);
            await this.walletServices.AddAsync(model);
            return 1;
        }
    }
}
